<?php
class eSIMAPI {
    private $apiUrl;
    private $accessCode;
    private $securityKey;
    
    public function __construct() {
        $this->apiUrl = ESIM_API_URL;
        $this->accessCode = ESIM_ACCESS_CODE;
        $this->securityKey = ESIM_SECURITY_KEY;
    }
    
    // Generate HMAC-SHA256 signature according to eSIM Access documentation
    // signData = Timestamp + RequestID + AccessCode + RequestBody
    // signature = HMACSHA256(signData, SecretKey)
    private function generateSignature($timestamp, $requestId, $requestBody) {
        $signData = $timestamp . $requestId . $this->accessCode . $requestBody;
        $signature = hash_hmac('sha256', $signData, $this->securityKey);
        return strtoupper($signature);
    }
    
    // Make API request
    private function makeRequest($endpoint, $method = 'POST', $data = null) {
        $url = $this->apiUrl . $endpoint;
        
        // Prepare request body
        $requestBody = '';
        if ($data && $method == 'POST') {
            $requestBody = json_encode($data);
        }
        
        // Generate authentication parameters
        $timestamp = (string)(round(microtime(true) * 1000)); // milliseconds
        $requestId = $this->generateUUID();
        $signature = $this->generateSignature($timestamp, $requestId, $requestBody);
        
        // Set headers according to eSIM Access API documentation
        $headers = [
            'Content-Type: application/json',
            'RT-Timestamp: ' . $timestamp,
            'RT-RequestID: ' . $requestId,
            'RT-AccessCode: ' . $this->accessCode,
            'RT-Signature: ' . $signature
        ];
        
        // Initialize cURL
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        if ($method == 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($requestBody) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $requestBody);
            }
        }
        
        // Execute request
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        // Log errors for debugging
        if ($httpCode < 200 || $httpCode >= 300) {
            error_log("eSIM API Error: HTTP $httpCode - URL: $url - Error: $error - Response: $response");
        }
        
        // Return parsed response
        if ($httpCode >= 200 && $httpCode < 300) {
            return json_decode($response, true);
        }
        
        return false;
    }
    
    // Generate UUID v4 for RT-RequestID
    private function generateUUID() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
    
    // Get all available packages
    public function getAllPackages() {
        $data = [
            'locationCode' => '',
            'type' => 'BASE',
            'packageCode' => '',
            'slug' => '',
            'iccid' => ''
        ];
        return $this->makeRequest('/open/package/list', 'POST', $data);
    }
    
    // Get packages for a specific country
    public function getCountryPackages($countryCode) {
        $data = [
            'locationCode' => $countryCode,
            'type' => 'BASE',
            'packageCode' => '',
            'slug' => '',
            'iccid' => ''
        ];
        return $this->makeRequest('/open/package/list', 'POST', $data);
    }
    
    // Get package by code
    public function getPackageByCode($packageCode) {
        $allPackages = $this->getAllPackages();
        
        if ($allPackages && isset($allPackages['obj']['packageList'])) {
            foreach ($allPackages['obj']['packageList'] as $package) {
                if ($package['packageCode'] == $packageCode || $package['slug'] == $packageCode) {
                    return $package;
                }
            }
        }
        
        return null;
    }
    
    // Create eSIM order
    public function createESIMOrder($packageCode, $quantity = 1) {
        return $this->makeRequest('/esim/order', [
            'packageCode' => $packageCode,
            'quantity' => $quantity
        ]);
    }
    
    // Get eSIM details
    public function getESIMDetails($esimId) {
        return $this->makeRequest('/esim/detail', [
            'esimId' => $esimId
        ]);
    }
    
    // Get order list
    public function getOrderList($page = 1, $pageSize = 20) {
        return $this->makeRequest('/esim/orderList', [
            'pageNum' => $page,
            'pageSize' => $pageSize
        ]);
    }
}

